#include <stdio.h>
#include <string.h>
#include "lexems.h"
#include "cfgstr.h"
#include <memuser.h>
#include <bddint.h>

#define NAMEBUFLEN 100

rec_mgr Mmgr_Node,
        Mmgr_Edge,
        Mmgr_Var,
        Mmgr_Oper,
        Mmgr_Varlist,
        Mmgr_Analysis,
        Mmgr_AllVars,
        Mmgr_WorkList,
        Mmgr_GdlCache;

VarType *variables;
OperType *opers;
CfgNode *nodes;
CfgNode *startnode, *endnode;
VarList *secretvars;
VarList *initkeys;
FILE *yyin;

int nextnodeid;
OperType *asgnopertype;

LexType l_result;
char l_idname[NAMEBUFLEN];
int l_idnamebuflen;
OperSort l_opsort;
int l_linenum;
int l_posnum;

int num_vars;
VarType **all_vars;

bdd_manager BDDM;
bdd TRUEBDD, FALSEBDD;

WorkList *startworklist, *endworklist;

void insert_worklist(CfgEdge *e)
{
  WorkList *nelem;
  
  if( startworklist == NULL )
  {
    nelem = (WorkList *)mem_new_rec(Mmgr_WorkList);
    nelem->edge = e;
    nelem->next = NULL;
    startworklist = endworklist = nelem;
    return;
  }
  
  nelem = startworklist;
  while( nelem != NULL )
  {
    if( nelem->edge == e )
      return;
    nelem = nelem->next;
  }
  
  nelem = (WorkList *)mem_new_rec(Mmgr_WorkList);
  nelem->edge = e;
  nelem->next = NULL;
  endworklist->next = nelem;
  endworklist = nelem;
}

CfgEdge *get_worklist(void)
{
  WorkList *nelem;
  CfgEdge *res;
  
  if( startworklist == NULL )
    return NULL;
  
  nelem = startworklist;
  startworklist = startworklist->next;
  if( startworklist == NULL )
    endworklist = NULL;
  res = nelem->edge;
  mem_free_rec(Mmgr_WorkList, nelem);
  return res;
}

void init_state(void)
{
  l_idnamebuflen = NAMEBUFLEN;
}

int intlength(int arg)
{
  int parg, res;
  
  parg = arg >= 0 ? arg : -arg;
  res = arg >= 0 ? 1 : 2;
  while( parg >= 10 )
  {
    res++;
    parg /= 10;
  }
  return res;
}

void delete_varlist(VarList *l)
{
  VarList *le;
  
  le = l;
  while(le != NULL )
  {
    l = le;
    le = le->next;
    mem_free_rec(Mmgr_Varlist, l);
  }
}

void adduniq_varlist(VarList **l, VarType *v)
{
  VarList *le;
  
  le = *l;
  while( le != NULL )
  {
    if( le->var == v )
      return;
    le = le->next;
  }
  le = (VarList *)mem_new_rec(Mmgr_Varlist);
  le->var = v;
  le->next = *l;
  *l = le;
}

void merge_varlist(VarList **dest, VarList *src)
{
  VarList *le;
  
  le = src;
  while( le != NULL )
  {
    adduniq_varlist(dest, le->var);
    le = le->next;
  }
}

bool is_in_varlist(VarType *v, VarList *vl)
{
  VarList *vlelem;
  
  vlelem = vl;
  while( vlelem != NULL )
  {
    if( vlelem->var == v )
      return TRUE;
    vlelem = vlelem->next;
  }
  return FALSE;
}

bool is_secretvar(VarType *v)
{
  return is_in_varlist(v, secretvars);
}

bool scan_variable( VarType **pvar )
{
  VarType *newvar, *vcur;
  
  if( (l_result != L_VARIABLE) && (l_result != L_CONSTANT) )
    return FALSE;
  
  newvar = NULL;
  vcur = variables;
  while( vcur != NULL )
  {
    if( (vcur->isConst == (l_result == L_CONSTANT) ) &&
        !vcur->isPseudo &&
        (strcmp(vcur->name, l_idname) == 0) )
    {
      newvar = vcur;
      break;
    }
    vcur = vcur->nextprim;
  }
  if( newvar == NULL )
  {
    newvar = (VarType *)mem_new_rec(Mmgr_Var);
    newvar->isConst = (l_result == L_CONSTANT);
    newvar->isPseudo = FALSE;
    newvar->name = (char *)mem_get_block((strlen(l_idname)+1)*sizeof(char));
    strcpy(newvar->name, l_idname);
    newvar->lpcfname = (char *)mem_get_block((strlen(l_idname)+4)*sizeof(char));
    sprintf(newvar->lpcfname, "(%s)E", l_idname);
    newvar->rpcfname = (char *)mem_get_block((strlen(l_idname)+4)*sizeof(char));
    sprintf(newvar->rpcfname, "[%s]E", l_idname);
    newvar->nextprim = variables;
    variables = newvar;
  }
  
  *pvar = newvar;
  yylex();
  return TRUE;
}

bool scan_operator( OperType **pvar )
{
  OperType *newoper, *vcur;
  
  if( l_result != L_FUNCTION )
    return FALSE;
  
  newoper = NULL;
  vcur = opers;
  while( vcur != NULL )
  {
    if( (vcur->sort == l_opsort) &&
        ( (l_opsort != O_NORMAL) ||
          (strcmp(l_idname, vcur->name) == 0) ) )
    {
      newoper = vcur;
      break;
    }
    vcur = vcur->nextprim;
  }
  if( newoper == NULL )
  {
    newoper = (OperType *)mem_new_rec(Mmgr_Oper);
    newoper->sort = l_opsort;
    if( l_opsort == O_NORMAL )
    {
      newoper->name = mem_get_block((strlen(l_idname)+1)*sizeof(char));
      strcpy(newoper->name, l_idname);
    }
    else if( l_opsort == O_ENC )
    {
      newoper->name = mem_get_block(4*sizeof(char));
      strcpy(newoper->name, "ENC");
    }
    else /* l_opsort == o_KEYGEN */
    {
      newoper->name = mem_get_block(8*sizeof(char));
      strcpy(newoper->name, "KEY_GEN");
    }
    newoper->nextprim = opers;
    opers = newoper;
  }
  
  *pvar = newoper;
  yylex();
  return TRUE;
}

bool scan_varlist(VarList **pvlist, bool allowConst)
{
  VarType *newvar;
  VarList *lelem, *llast;
  
  if( (l_result != L_VARIABLE) && (!allowConst || (l_result != L_CONSTANT)) )
  {
    *pvlist = NULL;
    return TRUE;
  }
  
  if( !scan_variable(&newvar) )
    return FALSE;
  lelem = (VarList *)mem_new_rec(Mmgr_Varlist);
  lelem->var = newvar;
  lelem->next = NULL;
  *pvlist = lelem;
  llast = lelem;
  
  while( l_result == L_COMMA )
  {
    yylex();
    if( (l_result != L_VARIABLE) && (!allowConst || (l_result != L_CONSTANT)) )
      return FALSE;
    if( !scan_variable(&newvar) )
      return FALSE;
    lelem = (VarList *)mem_new_rec(Mmgr_Varlist);
    lelem->var = newvar;
    lelem->next = NULL;
    llast->next = lelem;
    llast = lelem;
  }
  
  return TRUE;
}

bool scan_secrets(void)
{
  if( l_result != L_SECRETS )
    return TRUE;
  yylex();
  if( !scan_varlist(&secretvars, FALSE) )
    return FALSE;
  if( l_result != L_END )
    return FALSE;
  yylex();
  if( l_result != L_SECRETS )
    return FALSE;
  yylex();
  return TRUE;
}

bool scan_keys(void)
{
  if( l_result != L_KEYS )
    return TRUE;
  yylex();
  if( !scan_varlist(&initkeys, FALSE) )
    return FALSE;
  if( l_result != L_END )
    return FALSE;
  yylex();
  if( l_result != L_KEYS )
    return FALSE;
  yylex();
  return TRUE;
}

bool scan_stmts(CfgNode **firstnode, CfgNode **lastnode,
                                 VarList **changes, CfgNode *cdep);

bool scan_ifstmt(CfgNode **ifnode, CfgNode **mergenode,
                                   VarList **changes, CfgNode *cdep)
{
  CfgNode *thenstart, *thenend, *elsestart, *elseend;
  VarType *checkvar, *pseudovar;
  VarList *newchanges;
  CfgEdge *newedge;
  int nodeidlen;
  
  *ifnode = (CfgNode *)mem_new_rec(Mmgr_Node);
  nodeidlen = intlength(nextnodeid);
  (*ifnode)->id = nextnodeid++;
  (*ifnode)->ntype = N_BRANCH;

  pseudovar = mem_new_rec(Mmgr_Var);
  pseudovar->isConst = FALSE;
  pseudovar->isPseudo = TRUE;
  pseudovar->name = mem_get_block((nodeidlen+4)*sizeof(char));
  sprintf(pseudovar->name, "IF_%d",(*ifnode)->id);
  pseudovar->lpcfname = mem_get_block((nodeidlen+7)*sizeof(char));
  sprintf(pseudovar->lpcfname, "L(IF_%d)",(*ifnode)->id);
  pseudovar->rpcfname = mem_get_block((nodeidlen+7)*sizeof(char));
  sprintf(pseudovar->rpcfname, "R(IF_%d)",(*ifnode)->id);
  pseudovar->nextprim = variables;
  variables = pseudovar;
  (*ifnode)->specific.forBranch.pseudo = pseudovar;
  (*ifnode)->contrdep = cdep;
  (*ifnode)->edges = NULL;
  (*ifnode)->next = nodes;
  nodes = *ifnode;

  yylex();
  if( (l_result != L_VARIABLE) && (l_result != L_CONSTANT) )
    return FALSE;
  if( !scan_variable(&checkvar) )
    return FALSE;
  (*ifnode)->specific.forBranch.checked = checkvar;

  if( l_result != L_THEN )
    return FALSE;
  yylex();
  newchanges = NULL;
  if( !scan_stmts(&thenstart, &thenend, &newchanges, *ifnode) )
    return FALSE;

  if( l_result == L_ELSE )
  {
    yylex();
    if( !scan_stmts(&elsestart, &elseend, &newchanges, *ifnode) )
      return FALSE;
  }
  else
  {
    elsestart = elseend = NULL;
  }

  if( l_result != L_END )
    return FALSE;
  yylex();
  if( l_result != L_IF )
    return FALSE;
  yylex();

  *mergenode = (CfgNode *)mem_new_rec(Mmgr_Node);
  (*mergenode)->id = nextnodeid++;
  (*mergenode)->ntype = N_MERGE;
  (*mergenode)->specific.forMerge.corresp = *ifnode;
  (*mergenode)->specific.forMerge.changes = newchanges;
  (*mergenode)->contrdep = cdep;
  (*mergenode)->edges = NULL;
  (*mergenode)->next = nodes;
  nodes = *mergenode;
  merge_varlist(changes, newchanges);
  
  if( thenstart == NULL )
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_TRUE;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
  }
  else
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_TRUE;
    newedge->isbackedge = FALSE;
    newedge->target = thenstart;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_NORMAL;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = thenend->edges;
    thenend->edges = newedge;
  }
  
  if( elsestart == NULL )
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_FALSE;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
  }
  else
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_FALSE;
    newedge->isbackedge = FALSE;
    newedge->target = elsestart;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_NORMAL;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = elseend->edges;
    elseend->edges = newedge;
  }
  
  return TRUE;
}

bool scan_whilestmt(CfgNode **ifnode, CfgNode **mergenode,
                                   VarList **changes, CfgNode *cdep)
{
  CfgNode *loopstart, *loopend;
  VarType *checkvar, *pseudovar;
  VarList *newchanges;
  CfgEdge *newedge;
  int nodeidlen;
  
  *ifnode = (CfgNode *)mem_new_rec(Mmgr_Node);
  nodeidlen = intlength(nextnodeid);
  (*ifnode)->id = nextnodeid++;
  (*ifnode)->ntype = N_BRANCH;

  pseudovar = mem_new_rec(Mmgr_Var);
  pseudovar->isConst = FALSE;
  pseudovar->isPseudo = TRUE;
  pseudovar->name = mem_get_block((nodeidlen+4)*sizeof(char));
  sprintf(pseudovar->name, "IF_%d",(*ifnode)->id);
  pseudovar->lpcfname = mem_get_block((nodeidlen+7)*sizeof(char));
  sprintf(pseudovar->lpcfname, "L(IF_%d)",(*ifnode)->id);
  pseudovar->rpcfname = mem_get_block((nodeidlen+7)*sizeof(char));
  sprintf(pseudovar->rpcfname, "R(IF_%d)",(*ifnode)->id);
  pseudovar->nextprim = variables;
  variables = pseudovar;
  (*ifnode)->specific.forBranch.pseudo = pseudovar;
  (*ifnode)->contrdep = cdep;
  (*ifnode)->edges = NULL;
  (*ifnode)->next = nodes;
  nodes = *ifnode;

  yylex();
  if( (l_result != L_VARIABLE) && (l_result != L_CONSTANT) )
    return FALSE;
  if( !scan_variable(&checkvar) )
    return FALSE;
  (*ifnode)->specific.forBranch.checked = checkvar;

  if( l_result != L_DO )
    return FALSE;
  yylex();
  newchanges = NULL;
  if( !scan_stmts(&loopstart, &loopend, &newchanges, *ifnode) )
    return FALSE;

  if( l_result != L_END )
    return FALSE;
  yylex();
  if( l_result != L_WHILE )
    return FALSE;
  yylex();

  *mergenode = (CfgNode *)mem_new_rec(Mmgr_Node);
  (*mergenode)->id = nextnodeid++;
  (*mergenode)->ntype = N_MERGE;
  (*mergenode)->specific.forMerge.corresp = *ifnode;
  (*mergenode)->specific.forMerge.changes = newchanges;
  (*mergenode)->contrdep = cdep;
  (*mergenode)->edges = NULL;
  (*mergenode)->next = nodes;
  nodes = *mergenode;
  merge_varlist(changes, newchanges);
  
  if( loopstart == NULL )
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_TRUE;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
  }
  else
  {
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_TRUE;
    newedge->isbackedge = FALSE;
    newedge->target = loopstart;
    newedge->anres = NULL;
    newedge->next = (*ifnode)->edges;
    (*ifnode)->edges = newedge;
    newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
    newedge->etype = E_NORMAL;
    newedge->isbackedge = FALSE;
    newedge->target = *mergenode;
    newedge->anres = NULL;
    newedge->next = loopend->edges;
    loopend->edges = newedge;
  }
  
  newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
  newedge->etype = E_NORMAL;
  newedge->isbackedge = TRUE;
  newedge->target = *ifnode;
  newedge->anres = NULL;
  newedge->next = (*mergenode)->edges;
  (*mergenode)->edges = newedge;
  newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
  newedge->etype = E_FALSE;
  newedge->isbackedge = FALSE;
  newedge->target = *mergenode;
  newedge->anres = NULL;
  newedge->next = (*ifnode)->edges;
  (*ifnode)->edges = newedge;

  return TRUE;
}

bool scan_assign(CfgNode **newnode, VarList **changes, CfgNode *cdep)
{
  VarType *lhs, *rvar;
  OperType *oper;
  VarList *rhs;

  *newnode = (CfgNode *)mem_new_rec(Mmgr_Node);
  (*newnode)->id = nextnodeid++;
  (*newnode)->ntype = N_ASSIGN;
  (*newnode)->contrdep = cdep;
  (*newnode)->edges = NULL;
  (*newnode)->next = nodes;
  nodes = *newnode;
  
  if( !scan_variable(&lhs) )
    return FALSE;
  if( is_secretvar(lhs) )
  {
    printf("Assigning to the secret variable %s\n", lhs->name);
    return FALSE;
  }
  
  (*newnode)->specific.forAsgn.lhs = lhs;
  
  if( l_result != L_EQUALS )
    return FALSE;
  yylex();
  
  if( (l_result == L_VARIABLE) || (l_result == L_CONSTANT) )
  {
    if( !scan_variable(&rvar) )
      return FALSE;
    if( asgnopertype == NULL )
    {
      oper = (OperType *)mem_new_rec(Mmgr_Oper);
      oper->sort = O_ASSIGN;
      oper->name = NULL;
      oper->nextprim = opers;
      opers = oper;
      asgnopertype = oper;
    }
    (*newnode)->specific.forAsgn.oper = asgnopertype;
    rhs = (VarList *)mem_new_rec(Mmgr_Varlist);
    rhs->var = rvar;
    rhs->next = NULL;
    (*newnode)->specific.forAsgn.rhs = rhs;
  }
  else if( l_result == L_FUNCTION )
  {
    if( !scan_operator(&oper) )
      return FALSE;
    (*newnode)->specific.forAsgn.oper = oper;
    
    if( l_result != L_OPEN )
      return FALSE;
    yylex();
    
    if( !scan_varlist(&rhs, TRUE) )
      return FALSE;
    (*newnode)->specific.forAsgn.rhs = rhs;
    
    if( l_result != L_CLOSE )
      return FALSE;
    yylex();
  }
  else
    return FALSE;
  
  adduniq_varlist(changes, lhs);
  return TRUE;
}

bool scan_stmts(CfgNode **firstnode, CfgNode **lastnode,
                                 VarList **changes, CfgNode *cdep)
{
  CfgNode *newnode1, *newnode2;
  CfgEdge *newedge;
  
  *firstnode = *lastnode = NULL;
  while( (l_result == L_IF) ||
         (l_result == L_WHILE) ||
         (l_result == L_VARIABLE) )
  {
    if( l_result == L_IF )
    {
      if( !scan_ifstmt(&newnode1, &newnode2, changes, cdep) )
        return FALSE;
    }
    else if( l_result == L_WHILE )
    {
      if( !scan_whilestmt(&newnode1, &newnode2, changes, cdep) )
        return FALSE;
    }
    else /* l_result == L_VARIABLE */
    {
      if( !scan_assign(&newnode1, changes, cdep) )
        return FALSE;
      newnode2 = newnode1;
    }
    if( *firstnode == NULL )
    {
      *firstnode = newnode1;
      *lastnode = newnode2;
    }
    else
    {
      newedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
      newedge->etype = E_NORMAL;
      newedge->isbackedge = FALSE;
      newedge->target = newnode1;
      newedge->anres = NULL;
      newedge->next = (*lastnode)->edges;
      (*lastnode)->edges = newedge;
      *lastnode = newnode2;
    }
  }
  return TRUE;
}

bool scan_prog(void)
{
  VarList *changes;
  
  if( !scan_secrets() )
    return FALSE;
  if( !scan_keys() )
    return FALSE;
  changes = NULL;
  if( !scan_stmts(&startnode, &endnode, &changes, NULL) )
    return FALSE;
  delete_varlist(changes);
  return TRUE;
}

void write_variable(FILE *fp, VarType *v)
{
  bool sec;
  
  sec = is_secretvar(v);
  if( sec )
    fprintf(fp, "\fu%s\fn", v->name);
  else
    fprintf(fp, "%s", v->name);
}

void write_varlist(FILE *fp, VarList *vl)
{
  VarList *vlelem;
  
  if( vl == NULL )
    return;
  
  write_variable(fp, vl->var);
  vlelem = vl->next;
  while(vlelem != NULL )
  {
    fprintf(fp, ", ");
    write_variable(fp, vlelem->var);
    vlelem = vlelem->next;
  }
}

char *my_name_fun(bdd_manager dum1, bdd o, void *dum2)
{
  VarType *v;
  
  for( v = variables; v != NULL; v = v->nextprim )
  {
    if( v->pcf == o )
      return v->name;
    else if( v->lpcf == o )
      return v->lpcfname;
    else if( v->rpcf == o )
      return v->rpcfname;
  }
  return NULL;
}

typedef struct gdlcache_str {
  bdd f;
  int id;
  struct gdlcache_str *next;
} GdlCache;

int write_bddnodegdl(FILE *fp, bdd toprint, int edgeid, int bddid,
                     int *nodeid, int *tndid, int *fndid,
                      bdd mask, GdlCache **wrtn)
{
  int btype;
  bdd varbdd, checkbdd;
  bdd truebdd, falsebdd;
  int truenodeid, falsenodeid, mynodeid;
  GdlCache *cit;
  
  btype = bdd_type(BDDM, toprint);
  switch( btype )
  {
    case BDD_TYPE_ZERO:
    case BDD_TYPE_ONE:
      if( (btype == BDD_TYPE_ONE) && (*tndid != -1) )
        return *tndid;
      if( (btype == BDD_TYPE_ZERO) && (*fndid != -1) )
        return *fndid;
      fprintf(fp, "node: {\ntitle: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, *nodeid);
      fprintf(fp, "label: \"%d\"\n", btype == BDD_TYPE_ONE ? 1 : 0);
#ifdef BLACKWHITE
      fprintf(fp, "shape: circle\ncolor: white\n}\n\n");
#else
      fprintf(fp, "shape: circle\ncolor: %s\n}\n\n", btype == BDD_TYPE_ONE ? "green" : "red");
#endif
      mynodeid = *nodeid;
      (*nodeid)++;
      if( btype == BDD_TYPE_ONE )
        *tndid = mynodeid;
      else
        *fndid = mynodeid;
      break;
    default:
      varbdd = bdd_if(BDDM, toprint);
      truebdd = bdd_then(BDDM, toprint);
      falsebdd = bdd_else(BDDM, toprint);
      
      checkbdd = bdd_compose(BDDM, mask, varbdd, TRUEBDD);
      if( (checkbdd != TRUEBDD) && (truebdd == FALSEBDD) )
      {
        bdd_free(BDDM, checkbdd);
        return write_bddnodegdl(fp, falsebdd, edgeid, bddid, nodeid, tndid, fndid, mask, wrtn);
      }
      bdd_free(BDDM, checkbdd);
      
      for( cit = *wrtn; cit != NULL; cit = cit->next )
      {
        if( cit->f == BDD_POINTER(toprint) )
          return cit->id;
      }
      cit = (GdlCache *)mem_new_rec(Mmgr_GdlCache);
      cit->f = BDD_POINTER(toprint);
      cit->id = *nodeid;
      cit->next = *wrtn;
      *wrtn = cit;
      
      fprintf(fp, "node: {\ntitle: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, *nodeid);
      fprintf(fp, "label: \"%s\"\n", my_name_fun(BDDM, varbdd, NULL));
      fprintf(fp, "shape: box\ncolor: white\n}\n\n");
      mynodeid = *nodeid;
      (*nodeid)++;
      truenodeid = write_bddnodegdl(fp, truebdd, edgeid, bddid, nodeid, tndid, fndid, mask, wrtn);
      falsenodeid = write_bddnodegdl(fp, falsebdd, edgeid, bddid, nodeid, tndid, fndid, mask, wrtn);
      fprintf(fp, "node: {\ntitle: \"analres_%d_%d_res_%d_truelabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "label: \"1\"\nborderwidth: 1\nborderstyle: invisible\ncolor: white\n}\n\n");
      fprintf(fp, "node: {\ntitle: \"analres_%d_%d_res_%d_falselabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "label: \"0\"\nborderwidth: 1\nborderstyle: invisible\ncolor: white\n}\n\n");
      fprintf(fp, "leftnearedge: {\n");
      fprintf(fp, "sourcename: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "targetname: \"analres_%d_%d_res_%d_truelabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "arrowstyle: none\n}\n\n");
      fprintf(fp, "edge: {\n");
      fprintf(fp, "sourcename: \"analres_%d_%d_res_%d_truelabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "targetname: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, truenodeid);
      fprintf(fp, "}\n\n");
      fprintf(fp, "rightnearedge: {\n");
      fprintf(fp, "sourcename: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "targetname: \"analres_%d_%d_res_%d_falselabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "arrowstyle: none\n}\n\n");
      fprintf(fp, "edge: {\n");
      fprintf(fp, "sourcename: \"analres_%d_%d_res_%d_falselabel\"\n", edgeid, bddid, mynodeid);
      fprintf(fp, "targetname: \"analres_%d_%d_res_%d\"\n", edgeid, bddid, falsenodeid);
      fprintf(fp, "}\n\n");
  }
  return mynodeid;
}

void write_bddgdl(FILE *fp, bdd toprint, int edgeid, int bddid, bdd mask)
{
  int nodeid;
  GdlCache *topcache;
  int truenodeid, falsenodeid;
  
  fprintf(fp, "graph: {\ntitle: \"analres_%d_%d_res\"\nlabel: \"\"\n", edgeid, bddid);
#ifdef BLACKWHITE
  fprintf(fp, "color: white\n");
#else
  fprintf(fp, "color: pink\n");
#endif
  fprintf(fp, "layoutalgorithm: minbackward\nstatus: boxed\nmanhattan_edges: no\ncrossing_weight: median\nfinetuning: no\n\n");
  nodeid = 0;
  topcache = NULL;
  Mmgr_GdlCache = mem_new_rec_mgr(sizeof(GdlCache));
  truenodeid = falsenodeid = -1;
  write_bddnodegdl(fp, toprint, edgeid, bddid, &nodeid, &truenodeid, &falsenodeid, mask, &topcache);
  mem_free_rec_mgr(Mmgr_GdlCache);
  fprintf(fp, "}\n\n");
  
  fprintf(fp, "edge: {\n");
  fprintf(fp, "sourcename: \"analres_%d_%d\"\n", edgeid, bddid);
  fprintf(fp, "targetname: \"analres_%d_%d_res\"\n", edgeid, bddid);
  fprintf(fp, "}\n\n");
}

void write_analysis(FILE *fp, CfgEdge *edge, int edgeid)
{
  int i;
  int isfirst;
  bool *analkeys;
  CfgNode *cdep;
  
  bdd mask, temp;
  
  if( edge->anres == NULL )
  {
    fprintf(fp, "node: {\ntitle: \"analysis_%d\"\n", edgeid);
    fprintf(fp, "label: \"NO INFO\"\ncolor: yellow\n}\n\n");
    return;
  }
  
  fprintf(fp, "graph: {\ntitle: \"analysis_%d\"\n", edgeid);
  fprintf(fp, "color: white\nbordercolor: blue\nlabel: \"\"\nstate: boxed\n\n");
  
  fprintf(fp, "node: {\ntitle: \"analysis_%d_keys\"\n", edgeid);
  fprintf(fp, "color: yellow\n");
  fprintf(fp, "label: \"keys: ");
  analkeys = edge->anres->good_keys;
  isfirst = TRUE;
  for(i=0; i<num_vars; i++)
  {
    if( analkeys[i] )
    {
      if( !isfirst )
        fprintf(fp, ", ");
      isfirst = FALSE;
      fprintf(fp, "%s", all_vars[i]->name);
    }
  }
  fprintf(fp, "\"}\n\n");

  mask = bdd_identity(BDDM, FALSEBDD);
  for( i=0; i<num_vars; i++ )
    if( !all_vars[i]->isPseudo )
    {
      temp = mask;
      mask = bdd_or(BDDM, temp, all_vars[i]->pcf);
      bdd_free(BDDM, temp);
/*      if( edge->anres->good_keys[i] )  */
      {
        temp = mask;
        mask = bdd_or(BDDM, temp, all_vars[i]->rpcf);
        bdd_free(BDDM, temp);
      }
    }
  cdep = edge->target;
  if( cdep->ntype != N_MERGE )
  {
    if( !edge->isbackedge )
      cdep = cdep->contrdep;
  }
  else
  {
    cdep = cdep->specific.forMerge.corresp;
  }
  while( cdep != NULL )
  {
    temp = mask;
    mask = bdd_or(BDDM, temp, cdep->specific.forAsgn.lhs->pcf);
    bdd_free(BDDM, temp);
    cdep = cdep->contrdep;
  }

  fprintf(fp, "node: {\ntitle: \"analres_%d_0\"\nlabel: \"\fIBsecs\fn\"\ncolor: pink\n}\n\n", edgeid);
  fprintf(fp, "nearedge: {\n");
  fprintf(fp, "sourcename: \"analysis_%d_keys\"\n", edgeid);
  fprintf(fp, "targetname: \"analres_%d_0\"\n", edgeid);
  fprintf(fp, "arrowstyle: none\n}\n\n");
  write_bddgdl(fp, edge->anres->bsecs, edgeid, 0, mask);
  
  for( i=0; i<num_vars; i++ )
    if( edge->anres->good_keys[i] )
    {
      temp = mask;
      mask = bdd_or(BDDM, temp, all_vars[i]->lpcf);
      bdd_free(BDDM, temp);
    }

  fprintf(fp, "node: {\ntitle: \"analres_%d_1\"\nlabel: \"\fIBkeys\fn\"\ncolor: pink\n}\n\n", edgeid);
  fprintf(fp, "nearedge: {\n");
  fprintf(fp, "sourcename: \"analres_%d_0\"\n", edgeid);
  fprintf(fp, "targetname: \"analres_%d_1\"\n", edgeid);
  fprintf(fp, "arrowstyle: none\n}\n\n");
  write_bddgdl(fp, edge->anres->bkeys, edgeid, 1, mask);

  bdd_free(BDDM, mask);
  fprintf(fp, "}\n\n");
}

void write_gdl(char *filename)
{
  FILE *fp;
  CfgNode *cfgnode;
  CfgEdge *cfgedge;
  int edgeid;
  
  fp = fopen(filename, "w");
  edgeid = 0;
  
  fprintf(fp, "graph: {\ndisplay_edge_labels: yes\nnode.color: green\nnode.textmode: center\nmanhattan_edges: yes\nport_sharing: no\n\n");
  
  cfgnode = nodes;
  while( cfgnode != NULL )
  {
    fprintf(fp, "node: {\ntitle: \"node_%d\"\n", cfgnode->id);
    switch(cfgnode->ntype)
    {
      case N_ASSIGN:
        fprintf(fp, "shape: box\n");
        fprintf(fp, "label: \"%s := ", cfgnode->specific.forAsgn.lhs->name);
        if( cfgnode->specific.forAsgn.oper->sort == O_ASSIGN )
        {
          fprintf(fp, "%s", cfgnode->specific.forAsgn.rhs->var->name);
        }
        else
        {
          if( cfgnode->specific.forAsgn.oper->sort == O_NORMAL )
          {
            fprintf(fp, "%s(", cfgnode->specific.forAsgn.oper->name);
          }
          else
          {
            fprintf(fp, "\f02%s\f31(", cfgnode->specific.forAsgn.oper->name);
          }
          write_varlist(fp, cfgnode->specific.forAsgn.rhs);
          fprintf(fp, ")");
        }
        fprintf(fp, "\n\fI#%d\fn\"\n", cfgnode->id);          
        break;
      case N_BRANCH:
        fprintf(fp, "shape: rhomboid\n");
        fprintf(fp, "label: \"if ");
        write_variable(fp, cfgnode->specific.forBranch.checked);
        fprintf(fp, "\n\fI#%d\fn\"\n", cfgnode->id);
        break;
      case N_MERGE:
        fprintf(fp, "shape: uptrapezoid\n");
        fprintf(fp, "label: \"MERGE ");
        write_varlist(fp, cfgnode->specific.forMerge.changes);
        fprintf(fp, "\n\fI#%d\fn\"\n", cfgnode->id);
        break;
      case N_START:
        fprintf(fp, "shape: circle\n");
        fprintf(fp, "label: \"START\n\fI#%d\fn\"\n", cfgnode->id);
        break;
      case N_END:
        fprintf(fp, "shape: circle\n");
        fprintf(fp, "label: \"END\n\fI#%d\fn\"\n", cfgnode->id);
        break;
    }
    fprintf(fp, "}\n\n");

/*    
    if( cfgnode->ntype == N_MERGE )
    {
      fprintf(fp, "edge: {\n");
      fprintf(fp, "sourcename: \"node_%d\"\n", cfgnode->id);
      fprintf(fp, "targetname: \"node_%d\"\n", cfgnode->specific.forMerge.corresp->id);
      fprintf(fp, "priority: 0\narrowstyle: none\nlinestyle: dashed\nthickness: 1\n}\n\n");
    }
*/
    
    cfgedge = cfgnode->edges;
    while( cfgedge != NULL )
    {
      
      fprintf(fp, "node: {\n");
      fprintf(fp, "title: \"edge_%d\"\n", edgeid);
      fprintf(fp, "label: \"%s\"\n", cfgedge->etype == E_NORMAL ? "normal" : cfgedge->etype == E_TRUE ? "true" : "false");
      fprintf(fp, "color: white\n");
      fprintf(fp, "borderwidth: 1\n");
      fprintf(fp, "}\n\n");
      
      
      if( cfgedge->etype == E_NORMAL )
      {
        if( cfgedge->isbackedge )
          fprintf(fp, "backedge: {\n");
        else
          fprintf(fp, "edge: {\n");
      }
      else
      {
        fprintf(fp, "nearedge: {\n");
      }
      fprintf(fp, "sourcename: \"node_%d\"\ntargetname: \"edge_%d\"\n",
                     cfgnode->id, edgeid);
      fprintf(fp, "arrowstyle: none\n");
      fprintf(fp, "}\n\n");

      if( cfgedge->etype == E_NORMAL )
      {
        if( cfgedge->isbackedge )
          fprintf(fp, "backedge: {\n");
        else
          fprintf(fp, "edge: {\n");
      }
      else
      {
        fprintf(fp, "edge: {\n");
      }
      fprintf(fp, "sourcename: \"edge_%d\"\ntargetname: \"node_%d\"\n",
                     edgeid, cfgedge->target->id);
      fprintf(fp, "}\n\n");

      write_analysis(fp, cfgedge, edgeid);
      
      fprintf(fp, "nearedge: {\n");
      fprintf(fp, "sourcename: \"edge_%d\"\n", edgeid);
      fprintf(fp, "targetname: \"analysis_%d\"\n", edgeid);
      fprintf(fp, "}\n\n");

      edgeid++;
      cfgedge = cfgedge->next;
    }
    
    cfgnode = cfgnode->next;
  }
  
  fprintf(fp, "}\n");
  fclose(fp);
}

void makestartendnodes(void)
{
  CfgNode *cfgnode;
  CfgEdge *cfgedge;
  
  cfgnode = (CfgNode *)mem_new_rec(Mmgr_Node);
  cfgnode->id = nextnodeid++;
  cfgnode->ntype = N_START;
  cfgnode->contrdep = NULL;
  cfgedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
  cfgedge->etype = E_NORMAL;
  cfgedge->isbackedge = FALSE;
  cfgedge->anres = NULL;
  cfgedge->target = startnode;
  cfgedge->next = NULL;
  cfgnode->edges = cfgedge;
  cfgnode->next = nodes;
  nodes = cfgnode;
  startnode = cfgnode;

  cfgnode = (CfgNode *)mem_new_rec(Mmgr_Node);
  cfgnode->id = nextnodeid++;
  cfgnode->ntype = N_END;
  cfgnode->contrdep = NULL;
  cfgnode->edges = NULL;
  cfgnode->next = nodes;
  nodes = cfgnode;
  cfgedge = (CfgEdge *)mem_new_rec(Mmgr_Edge);
  cfgedge->etype = E_NORMAL;
  cfgedge->isbackedge = FALSE;
  cfgedge->target = cfgnode;
  cfgedge->anres = NULL;
  cfgedge->next = endnode->edges;
  endnode->edges = cfgedge;
  endnode = cfgnode;
}

void make_all_vars(void)
{
  VarType *v;
  int i;
  
  v = variables;
  num_vars = 0;
  while( v != NULL )
  {
    if( !v->isConst )
      num_vars++;
    v = v->nextprim;
  }

  Mmgr_AllVars = mem_new_rec_mgr(num_vars * sizeof(bool));
  all_vars = (VarType **)mem_get_block(num_vars * sizeof(VarType *));
  i = 0;
  v = variables;
  while( v != NULL )
  {
    if( !v->isConst )
    {
      all_vars[i] = v;
      v->index = i;
      i++;
    }
    v = v->nextprim;
  }
}

void init_analresult(void)
{
  bool *first_keys, *other_keys;
  VarType *v;
  VarList *vl;
  CfgNode *cfgnode;
  CfgEdge *cfgedge;
  int i;
  
  first_keys = (bool *)mem_new_rec(Mmgr_AllVars);
  other_keys = (bool *)mem_new_rec(Mmgr_AllVars);
  for(i=0; i<num_vars; i++)
    first_keys[i] = other_keys[i] = FALSE;

  v = variables;
  while( v != NULL )
  {
    if( !v->isConst &&
        !v->isPseudo )
      other_keys[v->index] = TRUE;
    v = v->nextprim;
  }
  
  vl = initkeys;
  while( vl != NULL )
  {
    first_keys[vl->var->index] = TRUE;
    vl = vl->next;
  }
  
  cfgnode = nodes;
  while( cfgnode != NULL )
  {
    cfgedge = cfgnode->edges;
    while( cfgedge != NULL )
    {
      cfgedge->anres = (AnalResult *)mem_new_rec(Mmgr_Analysis);
      cfgedge->anres->good_keys = (bool *)mem_new_rec(Mmgr_AllVars);
      cfgedge->anres->bkeys = NULL;
      cfgedge->anres->bsecs = NULL;
      if( cfgnode->ntype == N_START )
        memcpy(cfgedge->anres->good_keys, first_keys, num_vars * sizeof(bool));
      else
        memcpy(cfgedge->anres->good_keys, other_keys, num_vars * sizeof(bool));
      cfgedge = cfgedge->next;
    }
    cfgnode = cfgnode->next;
  }
  mem_free_rec(Mmgr_AllVars, first_keys);
  mem_free_rec(Mmgr_AllVars, other_keys);
}

void dfa( bool transfun(AnalResult *, CfgNode *, AnalResult *, EdgeType))
{
  CfgEdge *edge1, *edge2;
  CfgNode *over;

  startworklist = endworklist = NULL;
  edge1 = startnode->edges;
  while( edge1 != NULL )
  {
    insert_worklist(edge1);
    edge1 = edge1->next;
  }
  
  while( (edge1 = get_worklist()) != NULL )
  {
    over = edge1->target;
    edge2 = over->edges;
    while( edge2 != NULL )
    {
      if( transfun(edge1->anres, over, edge2->anres, edge2->etype) )
        insert_worklist(edge2);
      edge2 = edge2->next;
    }
  }
}

void add_one_other_var(bdd *mask, bdd vs, bdd v)
{
  bdd temp, sbst;
  
  sbst = bdd_compose(BDDM, vs, v, TRUEBDD);
  if( sbst != TRUEBDD )
  {
    temp = *mask;
    *mask = bdd_or(BDDM, temp, v);
    bdd_free(BDDM, temp);
  }
  bdd_free(BDDM, sbst);
}

bdd no_other_vars(bdd vs)
{
  int i;
  bdd mask, temp;
  
  mask = bdd_identity(BDDM, FALSEBDD);
  for( i=0; i<num_vars; i++ )
  {
    add_one_other_var(&mask, vs, all_vars[i]->pcf);
    add_one_other_var(&mask, vs, all_vars[i]->lpcf);
    add_one_other_var(&mask, vs, all_vars[i]->rpcf);
  }
  temp = mask;
  mask = bdd_not(BDDM, temp);
  bdd_free(BDDM, temp);
  return mask;
}

void do_masking(bool *good_keys, bdd oldkeys, bdd oldsecs, bdd *newkeys, bdd *newsecs, CfgNode *over)
{
  CfgNode *cdep;
  int i;
  bdd mask, mask2, temp;

  mask = bdd_identity(BDDM, FALSEBDD);
  for( i=0; i<num_vars; i++ )
    if( !all_vars[i]->isPseudo )
    {
      temp = mask;
      mask = bdd_or(BDDM, temp, all_vars[i]->pcf);
      bdd_free(BDDM, temp);
      if( good_keys[i] )
      {
        temp = mask;
        mask = bdd_or(BDDM, temp, all_vars[i]->rpcf);
        bdd_free(BDDM, temp);
      }
    }
  cdep = over->contrdep;

  while( cdep != NULL )
  {
    temp = mask;
    mask = bdd_or(BDDM, temp, cdep->specific.forAsgn.lhs->pcf);
    bdd_free(BDDM, temp);
    cdep = cdep->contrdep;
  }
  
  mask2 = no_other_vars(mask);
  *newsecs = bdd_and(BDDM, oldsecs, mask2);
  bdd_free(BDDM, mask2);
  
  for( i=0; i<num_vars; i++ )
    if( good_keys[i] )
    {
      temp = mask;
      mask = bdd_or(BDDM, temp, all_vars[i]->lpcf);
      bdd_free(BDDM, temp);
    }
  mask2 = no_other_vars(mask);
  *newkeys = bdd_and(BDDM, oldkeys, mask2);
  bdd_free(BDDM, mask2);
  bdd_free(BDDM, mask);
}

void join_flowres(AnalResult *toresult, AnalResult *newresult, bool *changes, CfgNode *over)
{
  bdd temp;
  int i;
  
  for(i=0; i<num_vars; i++)
  {
    if( toresult->good_keys[i] && !(newresult->good_keys[i]) )
      *changes = TRUE;
    toresult->good_keys[i] = toresult->good_keys[i] && newresult->good_keys[i];
  }

/*  
  do_masking(toresult->good_keys, newresult->bkeys, newresult->bsecs, &newkeys, &newsecs, over);
  bdd_free(BDDM, newresult->bkeys);
  bdd_free(BDDM, newresult->bsecs);
  newresult->bkeys = newkeys;
  newresult->bsecs = newsecs;
*/

  temp = bdd_and(BDDM, toresult->bkeys, newresult->bkeys);
  if( temp == toresult->bkeys )
    bdd_free(BDDM, temp);
  else
  {
    bdd_free(BDDM, toresult->bkeys);
    toresult->bkeys = temp;
    *changes = TRUE;
  }
  temp = bdd_and(BDDM, toresult->bsecs, newresult->bsecs);
  if( temp == toresult->bsecs )
    bdd_free(BDDM, temp);
  else
  {
    bdd_free(BDDM, toresult->bsecs);
    toresult->bsecs = temp;
    *changes = TRUE;
  }
}

bool flow_analysis(AnalResult *fromresult, CfgNode *over, AnalResult *toresult, EdgeType dummy)
{
  bool changes;
  AnalResult *newresult;
  int i;
  VarList *vl;
  
  changes = FALSE;
  newresult = (AnalResult *)mem_new_rec(Mmgr_Analysis);
  newresult->good_keys = (bool *)mem_new_rec(Mmgr_AllVars);
  switch( over->ntype )
  {
    case N_ASSIGN:
    {
      VarType *lhs;
      VarList *rhs;
      bdd *assoc;
      
      lhs = over->specific.forAsgn.lhs;
      rhs = over->specific.forAsgn.rhs;
      switch( over->specific.forAsgn.oper->sort )
      {
        case O_NORMAL:
        {
          bdd disj1, disj2, notxxe, conj;
          bdd temp;
          
          for(i=0; i<num_vars; i++)
            if( all_vars[i] == lhs )
              newresult->good_keys[i] = FALSE;
            else
              newresult->good_keys[i] = fromresult->good_keys[i];
          
          temp = bdd_or(BDDM, lhs->pcf, lhs->rpcf);
          notxxe = bdd_not(BDDM, temp);
          bdd_free(BDDM, temp);
          disj1 = bdd_and(BDDM, fromresult->bkeys, notxxe);
          assoc = (bdd *)mem_get_block(3 * sizeof(bdd));
          assoc[0] = lhs->pcf;
          assoc[1] = FALSEBDD;
          assoc[2] = NULL;
          bdd_temp_assoc(BDDM, assoc, TRUE);
          assoc[0] = lhs->rpcf;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
          assoc[1] = TRUEBDD;
          for( vl = rhs; vl != NULL; vl = vl->next )
            if( !vl->var->isConst )
            {
              assoc[0] = vl->var->pcf;
              bdd_augment_temp_assoc(BDDM, assoc, TRUE);
            }
          mem_free_block(assoc);
          bdd_assoc(BDDM, -1);
          disj2 = bdd_substitute(BDDM, fromresult->bkeys);
          conj = bdd_or(BDDM, disj1, disj2);
          bdd_free(BDDM, disj1);
          bdd_free(BDDM, disj2);
          temp = bdd_not(BDDM, lhs->lpcf);
          newresult->bkeys = bdd_and(BDDM, conj, temp);
          bdd_free(BDDM, conj);
          bdd_free(BDDM, temp);
          
          disj1 = bdd_and(BDDM, fromresult->bsecs, notxxe);
          bdd_free(BDDM, notxxe);
          disj2 = bdd_substitute(BDDM, fromresult->bsecs);
          newresult->bsecs = bdd_or(BDDM, disj1, disj2);
          bdd_free(BDDM, disj1);
          bdd_free(BDDM, disj2);
          
          break;
        }
        case O_ASSIGN:
        {
          bdd orl, orr, orv;
          
          for(i=0; i<num_vars; i++)
            if( all_vars[i] == lhs )
              newresult->good_keys[i] = fromresult->good_keys[rhs->var->index];
            else
              newresult->good_keys[i] = fromresult->good_keys[i];
          
          orl = bdd_or(BDDM, lhs->lpcf, rhs->var->lpcf);
          orr = bdd_or(BDDM, lhs->rpcf, rhs->var->rpcf);
          orv = bdd_or(BDDM, lhs->pcf, rhs->var->pcf);
          assoc = (bdd *)mem_get_block(13 * sizeof(bdd));
          assoc[0] = lhs->pcf;
          assoc[1] = FALSEBDD;
          assoc[2] = rhs->var->pcf;
          assoc[3] = orv;
          assoc[4] = lhs->rpcf;
          assoc[5] = FALSEBDD;
          assoc[6] = rhs->var->rpcf;
          assoc[7] = orr;
          assoc[8] = lhs->lpcf;
          assoc[9] = FALSEBDD;
          assoc[10] = rhs->var->lpcf;
          assoc[11] = orl;
          assoc[12] = NULL;
          bdd_temp_assoc(BDDM, assoc, TRUE);
          bdd_assoc(BDDM, -1);
          newresult->bkeys = bdd_substitute(BDDM, fromresult->bkeys);
          assoc[8] = NULL;
          bdd_temp_assoc(BDDM, assoc, TRUE);
          bdd_assoc(BDDM, -1);
          newresult->bsecs = bdd_substitute(BDDM, fromresult->bsecs);
          bdd_free(BDDM, orl);
          bdd_free(BDDM, orr);
          bdd_free(BDDM, orv);
          mem_free_block(assoc);
          
          break;
        }
        case O_ENC:
        {
          bdd disj1, disj2, disj3, disj31, disj32, notxxe, conj;
          bdd temp;
          int d2aid, faid;
          
          for(i=0; i<num_vars; i++)
            if( all_vars[i] == lhs )
              newresult->good_keys[i] = FALSE;
            else
              newresult->good_keys[i] = fromresult->good_keys[i];
          
          temp = bdd_or(BDDM, lhs->pcf, lhs->rpcf);
          notxxe = bdd_not(BDDM, temp);
          bdd_free(BDDM, temp);
          disj1 = bdd_and(BDDM, fromresult->bkeys, notxxe);
          assoc = (bdd *)mem_get_block(9 * sizeof(bdd));
          assoc[0] = lhs->pcf;
          assoc[1] = FALSEBDD;
          assoc[2] = lhs->rpcf;
          assoc[3] = FALSEBDD;
          assoc[4] = rhs->var->rpcf;
          assoc[5] = TRUEBDD;
          assoc[6] = rhs->next->var->pcf;
          assoc[7] = TRUEBDD;
          assoc[8] = NULL;
          d2aid = bdd_new_assoc(BDDM, assoc, TRUE);
          mem_free_block(assoc);
          bdd_assoc(BDDM, d2aid);
          disj2 = bdd_substitute(BDDM, fromresult->bkeys);
          conj = bdd_or(BDDM, disj1, disj2);
          if( fromresult->good_keys[rhs->var->index])
          {
            assoc = (bdd *)mem_get_block(5 * sizeof(bdd));
            assoc[0] = lhs->pcf;
            assoc[1] = FALSEBDD;
            assoc[2] = lhs->rpcf;
            assoc[3] = FALSEBDD;
            assoc[4] = NULL;
            faid = bdd_new_assoc(BDDM, assoc, TRUE);
            bdd_assoc(BDDM, faid);
            disj31 = bdd_substitute(BDDM, fromresult->bkeys);
            
            assoc[0] = lhs->pcf;
            assoc[1] = FALSEBDD;
            assoc[2] = rhs->next->var->pcf;
            assoc[3] = TRUEBDD;
            assoc[4] = NULL;
            bdd_temp_assoc(BDDM, assoc, TRUE);
            for(i=0; i< num_vars; i++)
              if( fromresult->good_keys[i] )
              {
                if( all_vars[i] == rhs->var )
                {
                  assoc[0] = all_vars[i]->lpcf;
                  assoc[1] = TRUEBDD;
                  assoc[2] = all_vars[i]->rpcf;
                  assoc[3] = FALSEBDD;
                  assoc[4] = NULL;
                  bdd_augment_temp_assoc(BDDM, assoc, TRUE);
                }
                else
                {
                  assoc[0] = all_vars[i]->lpcf;
                  assoc[1] = FALSEBDD;
                  assoc[2] = all_vars[i]->rpcf;
                  assoc[3] = all_vars[i]->lorrpcf;
                  assoc[4] = NULL;
                  bdd_augment_temp_assoc(BDDM, assoc, TRUE);
                }
              }
            assoc[0] = lhs->rpcf;
            assoc[1] = FALSEBDD;
            assoc[2] = NULL;
            bdd_augment_temp_assoc(BDDM, assoc, TRUE);
            mem_free_block(assoc);
            bdd_assoc(BDDM, -1);
            disj32 = bdd_substitute(BDDM, fromresult->bkeys);
            disj3 = bdd_and(BDDM, disj31, disj32);
            bdd_free(BDDM, disj31);
            bdd_free(BDDM, disj32);
          }
          else
            disj3 = bdd_identity(BDDM, FALSEBDD);
          temp = conj;
          conj = bdd_or(BDDM, temp, disj3);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, disj1);
          bdd_free(BDDM, disj2);
          bdd_free(BDDM, disj3);
          temp = bdd_not(BDDM, lhs->lpcf);
          newresult->bkeys = bdd_and(BDDM, conj, temp);
          bdd_free(BDDM, conj);
          bdd_free(BDDM, temp);
          
          disj1 = bdd_and(BDDM, fromresult->bsecs, notxxe);
          bdd_free(BDDM, notxxe);
          bdd_assoc(BDDM, d2aid);
          disj2 = bdd_substitute(BDDM, fromresult->bsecs);
          bdd_free_assoc(BDDM, d2aid);
          conj = bdd_or(BDDM, disj1, disj2);
          if( fromresult->good_keys[rhs->var->index] )
          {
            bdd_assoc(BDDM, faid);
            disj31 = bdd_substitute(BDDM, fromresult->bsecs);
            bdd_free_assoc(BDDM, faid);
            
            assoc = (bdd *)mem_get_block(7 * sizeof(bdd));
            assoc[0] = lhs->pcf;
            assoc[1] = FALSEBDD;
            assoc[2] = lhs->rpcf;
            assoc[3] = FALSEBDD;
            assoc[4] = rhs->next->var->pcf;
            assoc[5] = TRUEBDD;
            assoc[6] = NULL;
            bdd_temp_assoc(BDDM, assoc, TRUE);
            for(i=0; i< num_vars; i++)
              if( fromresult->good_keys[i] )
              {
                if( all_vars[i] == rhs->var )
                {
                  assoc[0] = all_vars[i]->lpcf;
                  assoc[1] = TRUEBDD;
                  assoc[2] = all_vars[i]->rpcf;
                  assoc[3] = FALSEBDD;
                  assoc[4] = NULL;
                  bdd_augment_temp_assoc(BDDM, assoc, TRUE);
                }
                else
                {
                  assoc[0] = all_vars[i]->lpcf;
                  assoc[1] = FALSEBDD;
                  assoc[2] = NULL;
                  bdd_augment_temp_assoc(BDDM, assoc, TRUE);
                }
              }
            for( vl = secretvars; vl != NULL; vl = vl->next )
            {
              assoc[0] = vl->var->pcf;
              assoc[1] = TRUEBDD;
              assoc[2] = NULL;
              bdd_augment_temp_assoc(BDDM, assoc, TRUE);
            }
            mem_free_block(assoc);
            bdd_assoc(BDDM, -1);
            disj32 = bdd_substitute(BDDM, fromresult->bkeys);
            disj3 = bdd_and(BDDM, disj31, disj32);
            bdd_free(BDDM, disj31);
            bdd_free(BDDM, disj32);
          }
          else
            disj3 = bdd_identity(BDDM, FALSEBDD);
          newresult->bsecs = bdd_or(BDDM, conj, disj3);
          bdd_free(BDDM, conj);
          bdd_free(BDDM, disj1);
          bdd_free(BDDM, disj2);
          bdd_free(BDDM, disj3);
          
          break;
        }
        case O_KEYGEN:
        {
          bdd disj1, disj2;
          bdd temp, temp2;
          int a1id, a2id;
          
          for(i=0; i<num_vars; i++)
            if( all_vars[i] == lhs )
              newresult->good_keys[i] = TRUE;
            else
              newresult->good_keys[i] = fromresult->good_keys[i];
          
          assoc = (bdd *)mem_get_block(5 * sizeof(bdd));
          assoc[0] = lhs->pcf;
          assoc[1] = FALSEBDD;
          assoc[2] = lhs->rpcf;
          assoc[3] = FALSEBDD;
          assoc[4] = NULL;
          a1id = bdd_new_assoc(BDDM, assoc, TRUE);
          assoc[0] = lhs->lpcf;
          a2id = bdd_new_assoc(BDDM, assoc, TRUE);
          mem_free_block(assoc);
          
          bdd_assoc(BDDM, a1id);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = bdd_not(BDDM, lhs->lpcf);
          disj1 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          bdd_assoc(BDDM, a2id);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = bdd_not(BDDM, lhs->pcf);
          disj2 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          newresult->bkeys = bdd_or(BDDM, disj1, disj2);
          bdd_free(BDDM, disj1);
          bdd_free(BDDM, disj2);
          
          bdd_assoc(BDDM, a1id);
          newresult->bsecs = bdd_substitute(BDDM, fromresult->bsecs);
          bdd_free_assoc(BDDM, a1id);
          bdd_free_assoc(BDDM, a2id);
          
          break;
        }
      }
      break;
    }
    case N_MERGE:
    {
      VarType *N;
      VarList *Z;
      bdd *assoc;
      
      bdd mnk1, mnk2, mns1, mns2;
      bdd B1, B2, B3, B4, B5;
      bdd temp, temp2, conj;
      
      int aix;
      
      N = over->specific.forMerge.corresp->specific.forAsgn.lhs;
      Z = over->specific.forMerge.changes;

      for( i=0; i<num_vars; i++ )
      {
        if( fromresult->good_keys[i] && is_in_varlist(all_vars[i], Z) )
        {
          assoc = (bdd *)mem_get_block(3 * sizeof(bdd));
          assoc[0] = N->pcf;
          assoc[1] = all_vars[i]->lpcf;
          assoc[2] = NULL;
          bdd_temp_assoc(BDDM, assoc, FALSE);
          bdd_assoc(BDDM, -1);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          newresult->good_keys[i] = !(temp == FALSEBDD);
          bdd_free(BDDM, temp);
          mem_free_block(assoc);
        }
        else
          newresult->good_keys[i] = fromresult->good_keys[i];
      }
      
      assoc = (bdd *)mem_get_block((6 * num_vars + 1) * sizeof(bdd));

      assoc[0] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      for( vl = Z; vl != NULL; vl = vl->next )
      {
        if( fromresult->good_keys[vl->var->index] )
        {
          assoc[0] = vl->var->lpcf;
          assoc[1] = FALSEBDD;
          assoc[2] = vl->var->rpcf;
          assoc[3] = FALSEBDD;
          assoc[4] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
        }
      }
      bdd_assoc(BDDM, -1);
      B1 = bdd_substitute(BDDM, fromresult->bkeys);
      
      assoc[0] = N->pcf;
      assoc[1] = TRUEBDD;
      assoc[2] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      for( i=0; i<num_vars; i++ )
        if( fromresult->good_keys[i] )
        {
          if( is_in_varlist(all_vars[i], Z) )
          {
            assoc[0] = all_vars[i]->lpcf;
            assoc[1] = all_vars[i]->lorrpcf;
            assoc[2] = all_vars[i]->rpcf;
            assoc[3] = FALSEBDD;
            assoc[4] = NULL;
          }
          else
          {
            assoc[0] = all_vars[i]->lpcf;
            assoc[1] = FALSEBDD;
            assoc[2] = all_vars[i]->rpcf;
            assoc[3] = all_vars[i]->lorrpcf;
            assoc[4] = NULL;
          }
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
        }
      bdd_assoc(BDDM, -1);
      B2 = bdd_substitute(BDDM, fromresult->bkeys);
      
      B3 = bdd_identity(BDDM, TRUEBDD);
      for( vl = Z; vl != NULL; vl = vl->next )
      {
        temp = bdd_not(BDDM, vl->var->pcf);
        temp2 = B3;
        B3 = bdd_and(BDDM, temp, temp2);
        bdd_free(BDDM, temp);
        bdd_free(BDDM, temp2);
        if( !fromresult->good_keys[vl->var->index] )
        {
          temp = bdd_not(BDDM, vl->var->rpcf);
          temp2 = B3;
          B3 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
        }
      }
      
      B4 = bdd_identity(BDDM, TRUEBDD);
      aix = 0;
      for( i=0; i<num_vars; i++ )
      {
        assoc[aix++] = all_vars[i]->pcf;
        assoc[aix++] = FALSEBDD;
        assoc[aix++] = all_vars[i]->rpcf;
        assoc[aix++] = FALSEBDD;
        if( fromresult->good_keys[i] )
        {
          if( !is_in_varlist(all_vars[i], Z) )
          {
            assoc[aix++] = all_vars[i]->lpcf;
            assoc[aix++] = FALSEBDD;
          }
          else
          {
            assoc[aix++] = all_vars[i]->lpcf;
            assoc[aix++] = all_vars[i]->lorrpcf;
          }
        }
      }
      assoc[aix] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      aix = 0;
      for( i=0; i<num_vars; i++ )
        if( fromresult->good_keys[i] && is_in_varlist(all_vars[i], Z) )
        {
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = FALSEBDD;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = all_vars[i]->lorrpcf;
          assoc[aix++] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
          bdd_assoc(BDDM, -1);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = B4;
          B4 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          aix = 0;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = FALSEBDD;
        }
      
      B5 = bdd_identity(BDDM, TRUEBDD);
      aix = 0;
      for( i=0; i<num_vars; i++ )
      {
        assoc[aix++] = all_vars[i]->pcf;
        assoc[aix++] = FALSEBDD;
        assoc[aix++] = all_vars[i]->rpcf;
        assoc[aix++] = FALSEBDD;
        assoc[aix++] = all_vars[i]->lpcf;
        assoc[aix++] = FALSEBDD;
      }
      assoc[aix] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      aix = 0;
      for( i=0; i<num_vars; i++ )
        if( fromresult->good_keys[i] && is_in_varlist(all_vars[i], Z) )
        {
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = all_vars[i]->landrpcf;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = all_vars[i]->landrpcf;
          assoc[aix++] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
          bdd_assoc(BDDM, -1);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = B5;
          B5 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          aix = 0;
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = FALSEBDD;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = FALSEBDD;
        }
      
      temp = bdd_and(BDDM, B1, B2);
      temp2 = bdd_and(BDDM, temp, B3);
      bdd_free(BDDM, temp);
      temp = bdd_and(BDDM, temp2, B4);
      bdd_free(BDDM, temp2);
      mnk1 = bdd_and(BDDM, temp, B5);
      bdd_free(BDDM, temp);
      bdd_free(BDDM, B1);
      bdd_free(BDDM, B2);
      bdd_free(BDDM, B4);
      bdd_free(BDDM, B5);
      
      assoc[0] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      for( vl = Z; vl != NULL; vl = vl->next )
      {
        if( fromresult->good_keys[vl->var->index] )
        {
          assoc[0] = vl->var->rpcf;
          assoc[1] = FALSEBDD;
          assoc[2] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
        }
      }
      bdd_assoc(BDDM, -1);
      B1 = bdd_substitute(BDDM, fromresult->bsecs);
      
      assoc[0] = N->pcf;
      assoc[1] = TRUEBDD;
      assoc[2] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      for( i=0; i<num_vars; i++ )
      {
        if( fromresult->good_keys[i] )
        {
          if( is_in_varlist(all_vars[i], Z) )
          {
            assoc[0] = all_vars[i]->lpcf;
            assoc[1] = all_vars[i]->rpcf;
            assoc[2] = all_vars[i]->rpcf;
            assoc[3] = FALSEBDD;
            assoc[4] = NULL;
          }
          else
          {
            assoc[0] = all_vars[i]->lpcf;
            assoc[1] = FALSEBDD;
            assoc[2] = NULL;
          }
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
        }
        if( is_secretvar(all_vars[i]) )
        {
          assoc[0] = all_vars[i]->pcf;
          assoc[1] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, FALSE);
        }
      }
      bdd_assoc(BDDM, -1);
      B2 = bdd_substitute(BDDM, fromresult->bkeys);
      
      B4 = bdd_identity(BDDM, TRUEBDD);
      aix = 0;
      for( i=0; i<num_vars; i++ )
      {
        assoc[aix++] = all_vars[i]->pcf;
        assoc[aix++] = FALSEBDD;
        assoc[aix++] = all_vars[i]->rpcf;
        assoc[aix++] = FALSEBDD;
        if( fromresult->good_keys[i] )
        {
          if( !is_in_varlist(all_vars[i], Z) )
          {
            assoc[aix++] = all_vars[i]->lpcf;
            assoc[aix++] = FALSEBDD;
          }
          else
          {
            assoc[aix++] = all_vars[i]->lpcf;
            assoc[aix++] = all_vars[i]->rpcf;
          }
        }
      }
      assoc[aix] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      aix = 0;
      for( i=0; i<num_vars; i++ )
        if( fromresult->good_keys[i] && is_in_varlist(all_vars[i], Z) )
        {
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = FALSEBDD;
          assoc[aix++] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
          bdd_assoc(BDDM, -1);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = B4;
          B4 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          aix = 0;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = FALSEBDD;
        }
      
      temp = bdd_and(BDDM, B1, B2);
      temp2 = bdd_and(BDDM, temp, B3);
      bdd_free(BDDM, temp);
      mns1 = bdd_and(BDDM, temp2, B4);
      bdd_free(BDDM, temp2);
      bdd_free(BDDM, B1);
      bdd_free(BDDM, B2);
      bdd_free(BDDM, B3);
      bdd_free(BDDM, B4);
      
      B1 = bdd_compose(BDDM, fromresult->bkeys, N->pcf, TRUEBDD);
      
      aix = 0;
      for( i=0; i<num_vars; i++ )
      {
        assoc[aix++] = all_vars[i]->pcf;
        assoc[aix++] = FALSEBDD;
        if( fromresult->good_keys[i] )
        {
          if( is_in_varlist(all_vars[i], Z) )
          {
            assoc[aix++] = all_vars[i]->lpcf;
            assoc[aix++] = FALSEBDD;
            assoc[aix++] = all_vars[i]->rpcf;
            assoc[aix++] = all_vars[i]->lpcf;
          }
          else
          {
            assoc[aix++] = all_vars[i]->rpcf;
            assoc[aix++] = FALSEBDD;
          }
        }
        else
        {
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = FALSEBDD;
        }
      }
      assoc[aix] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      bdd_assoc(BDDM, -1);
      B2 = bdd_substitute(BDDM, fromresult->bkeys);
      
      B3 = bdd_identity(BDDM, TRUEBDD);
      aix = 0;
      for( i=0; i<num_vars; i++ )
      {
        assoc[aix++] = all_vars[i]->pcf;
        assoc[aix++] = FALSEBDD;
        assoc[aix++] = all_vars[i]->rpcf;
        assoc[aix++] = FALSEBDD;
        if( fromresult->good_keys[i] && !is_in_varlist(all_vars[i], Z) )
        {
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = FALSEBDD;
        }
      }
      assoc[aix] = NULL;
      bdd_temp_assoc(BDDM, assoc, TRUE);
      aix = 0;
      for( i=0; i< num_vars; i++ )
        if( fromresult->good_keys[i] && is_in_varlist(all_vars[i], Z) )
        {
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix++] = FALSEBDD;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = all_vars[i]->lpcf;
          assoc[aix] = NULL;
          bdd_augment_temp_assoc(BDDM, assoc, TRUE);
          temp = bdd_substitute(BDDM, fromresult->bkeys);
          temp2 = B3;
          B3 = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
          aix = 0;
          assoc[aix++] = all_vars[i]->rpcf;
          assoc[aix++] = FALSEBDD;
        }
      
      temp = bdd_and(BDDM, B1, B2);
      mnk2 = bdd_and(BDDM, temp, B3);
      bdd_free(BDDM, temp);
      bdd_free(BDDM, B1);
      bdd_free(BDDM, B2);
      bdd_free(BDDM, B3);
      
      mns2 = bdd_compose(BDDM, fromresult->bsecs, N->pcf, TRUEBDD);
      
      newresult->bsecs = bdd_or(BDDM, mns1, mns2);
      bdd_free(BDDM, mns1);
      bdd_free(BDDM, mns2);
      
      conj = bdd_identity(BDDM, TRUEBDD);
      for( i=0; i< num_vars; i++ )
        if( fromresult->good_keys[i] && !newresult->good_keys[i] )
        {
          temp = bdd_not(BDDM, all_vars[i]->lpcf);
          temp2 = conj;
          conj = bdd_and(BDDM, temp, temp2);
          bdd_free(BDDM, temp);
          bdd_free(BDDM, temp2);
        }
      temp = bdd_or(BDDM, mnk1, mnk2);
      newresult->bkeys = bdd_and(BDDM, temp, conj);
      bdd_free(BDDM, mnk1);
      bdd_free(BDDM, mnk2);
      bdd_free(BDDM, temp);
      bdd_free(BDDM, conj);
      
      break;
    }
    case N_BRANCH:
    case N_START:
    case N_END:
  }
  join_flowres(toresult, newresult, &changes, over);
  bdd_free(BDDM, newresult->bkeys);
  bdd_free(BDDM, newresult->bsecs);
  mem_free_rec(Mmgr_AllVars, newresult->good_keys);
  mem_free_rec(Mmgr_Analysis, newresult);
  return changes;
}

void init_mainanal(void)
{
  CfgNode *cfgnode;
  CfgEdge *cfgedge;
  VarType *v;
  VarList *vl;
  int i;

  BDDM = bdd_init();
  bdd_dynamic_reordering(BDDM, bdd_reorder_sift);
  TRUEBDD = bdd_one(BDDM);
  FALSEBDD = bdd_zero(BDDM);
  
  for( v = variables; v != NULL; v = v->nextprim )
  {
    if( !v->isConst )
    {
      v->pcf = bdd_new_var_last(BDDM);
      v->lpcf = bdd_new_var_last(BDDM);
      v->rpcf = bdd_new_var_last(BDDM);
      v->lorrpcf = bdd_or(BDDM, v->lpcf, v->rpcf);
      v->landrpcf = bdd_and(BDDM, v->lpcf, v->rpcf);
    }
  }
  
  for( cfgnode = nodes; cfgnode != NULL; cfgnode = cfgnode->next )
  {
    for( cfgedge = cfgnode->edges; cfgedge != NULL; cfgedge = cfgedge->next )
    {
      if( cfgnode == startnode )
      {
        bdd temp, temp2, temp3;
        
        temp = bdd_identity(BDDM, FALSEBDD);
        for( vl = secretvars; vl != NULL; vl = vl->next )
        {
          temp2 = temp;
          temp = bdd_or(BDDM, temp2, vl->var->pcf);
          bdd_free(BDDM, temp2);
          temp2 = temp;
          temp = bdd_or(BDDM, temp2, vl->var->rpcf);
          bdd_free(BDDM, temp2);
        }
        temp2 = bdd_not(BDDM, temp);
        cfgedge->anres->bsecs = temp2;
        bdd_free(BDDM, temp);

        temp = bdd_identity(BDDM, FALSEBDD);
        for(i=0; i<num_vars; i++)
          if( cfgedge->anres->good_keys[i] )
          {
            temp2 = temp;
            temp3 = bdd_and(BDDM, all_vars[i]->lpcf, all_vars[i]->pcf);
            temp = bdd_or(BDDM, temp2, temp3);
            bdd_free(BDDM, temp2);
            bdd_free(BDDM, temp3);
          }
          else
          {
            temp2 = temp;
            temp = bdd_or(BDDM, temp2, all_vars[i]->lpcf);
            bdd_free(BDDM, temp2);
          }
        temp2 = bdd_not(BDDM, temp);
        cfgedge->anres->bkeys = temp2;
        bdd_free(BDDM, temp);
      }
      else
      {
        cfgedge->anres->bkeys = bdd_identity(BDDM, TRUEBDD);
        cfgedge->anres->bsecs = bdd_identity(BDDM, TRUEBDD);
      }
    }
  }
}

void change_ifs(void)
{
  CfgNode *n1, *n2;
  CfgEdge *ce;
  OperType *phiop, *constop;
  VarList *vl1;
  VarType *psvar;
  
  phiop = (OperType *)mem_new_rec(Mmgr_Oper);
  phiop->sort = O_ASSIGN;
  phiop->name = NULL;
  phiop->nextprim = opers;
  opers = phiop;
  constop = (OperType *)mem_new_rec(Mmgr_Oper);
  constop->sort = O_NORMAL;
  constop->name = (char *)mem_get_block(5 * sizeof(char));
  sprintf(constop->name, "TRUE");
  constop->nextprim = opers;
  opers = constop;

  for( n1 = nodes; n1 != NULL; n1 = n1->next )
  {
    for( ce = n1->edges; ce != NULL; ce = ce->next )
    {
      n2 = ce->target;
      if( !ce->isbackedge && (n2->ntype == N_BRANCH) )
      {
        psvar = n2->specific.forBranch.pseudo;
/*
        nnew = (CfgNode *)mem_new_rec(Mmgr_Node);
        nnew->id = nextnodeid++;
        nnew->ntype = N_ASSIGN;
        nnew->specific.forAsgn.lhs = psvar;
        nnew->specific.forAsgn.oper = constop;
        nnew->specific.forAsgn.rhs = NULL;
        nnew->contrdep = n2->contrdep;
        nnew->edges = (CfgEdge *)mem_new_rec(Mmgr_Edge);
        nnew->edges->etype = E_NORMAL;
        nnew->edges->isbackedge = FALSE;
        nnew->edges->target = n2;
        nnew->edges->next = NULL;
        nnew->edges->anres = NULL;
        nnew->next = nodes;
        nodes = nnew;
        ce->target = nnew;
*/        
        vl1 = (VarList *)mem_new_rec(Mmgr_Varlist);
        vl1->var = n2->specific.forBranch.checked;
        vl1->next = NULL;
        n2->ntype = N_ASSIGN;
        n2->specific.forAsgn.lhs = psvar;
        n2->specific.forAsgn.oper = phiop;
        n2->specific.forAsgn.rhs = vl1;
      }
    }
  }
}

int main(int argc, char *argv[])
{
  bool goodparse;
  char *gdlfilename;
  
  Mmgr_Node = mem_new_rec_mgr(sizeof(CfgNode));
  Mmgr_Edge = mem_new_rec_mgr(sizeof(CfgEdge));
  Mmgr_Var = mem_new_rec_mgr(sizeof(VarType));
  Mmgr_Oper = mem_new_rec_mgr(sizeof(OperType));
  Mmgr_Varlist = mem_new_rec_mgr(sizeof(VarList));
  Mmgr_Analysis = mem_new_rec_mgr(sizeof(AnalResult));
  Mmgr_WorkList = mem_new_rec_mgr(sizeof(WorkList));
  
  if( argc < 2 )
  {
    printf("Usage: %s <filename>\n", argv[0]);
    return 1;
  }
  init_state();
  variables = NULL;
  opers = NULL;
  nodes = NULL;
  secretvars = initkeys = NULL;
  nextnodeid = 1;
  asgnopertype = NULL;
  yyin = fopen(argv[1], "r");
  yylex();
  goodparse = scan_prog();
  fclose(yyin);
  if( !goodparse )
  {
    printf("Bad Parse before line %d position %d!\n", l_linenum, l_posnum);
    return 1;
  }

  makestartendnodes();

  gdlfilename = (char *)mem_get_block((strlen(argv[1])+7)*sizeof(char));
  sprintf(gdlfilename, "%s.1.gdl", argv[1]);
  write_gdl(gdlfilename);

  make_all_vars();
  change_ifs();
  init_analresult();
  init_mainanal();

  dfa(flow_analysis);

  sprintf(gdlfilename, "%s.2.gdl", argv[1]);
  bdd_reorder(BDDM);
  write_gdl(gdlfilename);

  return 0;
}

void expose_print(bdd f)
{
  bdd_print_bdd(BDDM, f, my_name_fun, bdd_terminal_id_fn_none, NULL, stdout);
}
